/* --------------------------------------------------------------------------------------------
 * Copyright (c) Remy Suen. All rights reserved.
 * Licensed under the MIT License. See License.txt in the project root for license information.
 * ------------------------------------------------------------------------------------------ */
'use strict';
Object.defineProperty(exports, "__esModule", { value: true });
exports.DockerHighlight = void 0;
var vscode_languageserver_textdocument_1 = require("vscode-languageserver-textdocument");
var vscode_languageserver_types_1 = require("vscode-languageserver-types");
var dockerfile_ast_1 = require("dockerfile-ast");
var dockerDefinition_1 = require("./dockerDefinition");
var docker_1 = require("./docker");
var DockerHighlight = /** @class */ (function () {
    function DockerHighlight() {
    }
    DockerHighlight.prototype.computeHighlightRanges = function (content, position) {
        var dockerfile = dockerfile_ast_1.DockerfileParser.parse(content);
        var provider = new dockerDefinition_1.DockerDefinition();
        var definitionRange = provider.computeDefinitionRange(content, position);
        var image = definitionRange === null ? dockerfile.getContainingImage(position) : dockerfile.getContainingImage(definitionRange.start);
        var highlights = [];
        if (definitionRange === null) {
            for (var _i = 0, _a = dockerfile.getCOPYs(); _i < _a.length; _i++) {
                var instruction = _a[_i];
                var flag = instruction.getFromFlag();
                if (flag) {
                    var range = flag.getValueRange();
                    if (range && range.start.line === position.line && range.start.character <= position.character && position.character <= range.end.character) {
                        var stage = flag.getValue();
                        for (var _b = 0, _c = dockerfile.getCOPYs(); _b < _c.length; _b++) {
                            var other = _c[_b];
                            var otherFlag = other.getFromFlag();
                            if (otherFlag && otherFlag.getValue().toLowerCase() === stage.toLowerCase()) {
                                highlights.push(vscode_languageserver_types_1.DocumentHighlight.create(otherFlag.getValueRange(), vscode_languageserver_types_1.DocumentHighlightKind.Read));
                            }
                        }
                        return highlights;
                    }
                }
            }
            for (var _d = 0, _e = dockerfile.getFROMs(); _d < _e.length; _d++) {
                var from = _e[_d];
                for (var _f = 0, _g = from.getVariables(); _f < _g.length; _f++) {
                    var variable = _g[_f];
                    if (docker_1.Util.isInsideRange(position, variable.getNameRange())) {
                        var name = variable.getName();
                        for (var _h = 0, _j = dockerfile.getFROMs(); _h < _j.length; _h++) {
                            var loopFrom = _j[_h];
                            for (var _k = 0, _l = loopFrom.getVariables(); _k < _l.length; _k++) {
                                var fromVariable = _l[_k];
                                if (fromVariable.getName() === name) {
                                    highlights.push(vscode_languageserver_types_1.DocumentHighlight.create(fromVariable.getNameRange(), vscode_languageserver_types_1.DocumentHighlightKind.Read));
                                }
                            }
                        }
                        return highlights;
                    }
                }
            }
            if (image !== null) {
                for (var _m = 0, _o = image.getInstructions(); _m < _o.length; _m++) {
                    var instruction = _o[_m];
                    for (var _p = 0, _q = instruction.getVariables(); _p < _q.length; _p++) {
                        var variable = _q[_p];
                        if (docker_1.Util.isInsideRange(position, variable.getNameRange())) {
                            var name = variable.getName();
                            for (var _r = 0, _s = image.getInstructions(); _r < _s.length; _r++) {
                                var instruction_1 = _s[_r];
                                if (!(instruction_1 instanceof dockerfile_ast_1.From)) {
                                    for (var _t = 0, _u = instruction_1.getVariables(); _t < _u.length; _t++) {
                                        var variable_1 = _u[_t];
                                        if (variable_1.getName() === name) {
                                            highlights.push(vscode_languageserver_types_1.DocumentHighlight.create(variable_1.getNameRange(), vscode_languageserver_types_1.DocumentHighlightKind.Read));
                                        }
                                    }
                                }
                            }
                            return highlights;
                        }
                    }
                }
            }
        }
        else {
            for (var _v = 0, _w = dockerfile.getInstructions(); _v < _w.length; _v++) {
                var instruction = _w[_v];
                if (instruction instanceof dockerfile_ast_1.Copy || instruction instanceof dockerfile_ast_1.Run) {
                    for (var _x = 0, _y = instruction.getHeredocs(); _x < _y.length; _x++) {
                        var heredoc = _y[_x];
                        var nameRange = heredoc.getNameRange();
                        if (docker_1.Util.positionEquals(definitionRange.start, nameRange.start) && docker_1.Util.positionEquals(definitionRange.start, nameRange.start)) {
                            highlights.push(vscode_languageserver_types_1.DocumentHighlight.create(definitionRange, vscode_languageserver_types_1.DocumentHighlightKind.Write));
                            var delimiterRange = heredoc.getDelimiterRange();
                            if (delimiterRange !== null) {
                                highlights.push(vscode_languageserver_types_1.DocumentHighlight.create(delimiterRange, vscode_languageserver_types_1.DocumentHighlightKind.Read));
                            }
                            return highlights;
                        }
                    }
                }
            }
            var document = vscode_languageserver_textdocument_1.TextDocument.create("", "", 0, content);
            var definition = document.getText().substring(document.offsetAt(definitionRange.start), document.offsetAt(definitionRange.end));
            var isBuildStage = false;
            var fromInstructions = dockerfile.getFROMs();
            for (var _z = 0, fromInstructions_1 = fromInstructions; _z < fromInstructions_1.length; _z++) {
                var from = fromInstructions_1[_z];
                var stage = from.getBuildStage();
                if (stage && definition.toLowerCase() === stage.toLowerCase()) {
                    highlights.push(vscode_languageserver_types_1.DocumentHighlight.create(from.getBuildStageRange(), vscode_languageserver_types_1.DocumentHighlightKind.Write));
                    isBuildStage = true;
                }
            }
            if (isBuildStage) {
                for (var _0 = 0, fromInstructions_2 = fromInstructions; _0 < fromInstructions_2.length; _0++) {
                    var from = fromInstructions_2[_0];
                    if (from.getRange().start.line > definitionRange.start.line) {
                        if (from.getImageName() === definition) {
                            highlights.push(vscode_languageserver_types_1.DocumentHighlight.create(from.getImageNameRange(), vscode_languageserver_types_1.DocumentHighlightKind.Read));
                        }
                    }
                }
                for (var _1 = 0, _2 = dockerfile.getCOPYs(); _1 < _2.length; _1++) {
                    var instruction = _2[_1];
                    var flag = instruction.getFromFlag();
                    if (flag) {
                        if (flag.getValue().toLowerCase() === definition.toLowerCase()) {
                            highlights.push(vscode_languageserver_types_1.DocumentHighlight.create(flag.getValueRange(), vscode_languageserver_types_1.DocumentHighlightKind.Read));
                        }
                    }
                }
                return highlights;
            }
            for (var _3 = 0, _4 = image.getARGs(); _3 < _4.length; _3++) {
                var arg = _4[_3];
                var property = arg.getProperty();
                // property may be null if it's an ARG with no arguments
                if (property && property.getName() === definition) {
                    highlights.push(vscode_languageserver_types_1.DocumentHighlight.create(property.getNameRange(), vscode_languageserver_types_1.DocumentHighlightKind.Write));
                }
            }
            for (var _5 = 0, _6 = image.getENVs(); _5 < _6.length; _5++) {
                var env = _6[_5];
                for (var _7 = 0, _8 = env.getProperties(); _7 < _8.length; _7++) {
                    var property = _8[_7];
                    if (property.getName() === definition) {
                        highlights.push(vscode_languageserver_types_1.DocumentHighlight.create(property.getNameRange(), vscode_languageserver_types_1.DocumentHighlightKind.Write));
                    }
                }
            }
            for (var _9 = 0, _10 = image.getInstructions(); _9 < _10.length; _9++) {
                var instruction = _10[_9];
                // only highlight variables in non-FROM instructions
                if (!(instruction instanceof dockerfile_ast_1.From)) {
                    for (var _11 = 0, _12 = instruction.getVariables(); _11 < _12.length; _11++) {
                        var variable = _12[_11];
                        if (variable.getName() === definition) {
                            highlights.push(vscode_languageserver_types_1.DocumentHighlight.create(variable.getNameRange(), vscode_languageserver_types_1.DocumentHighlightKind.Read));
                        }
                    }
                }
            }
            for (var _13 = 0, _14 = dockerfile.getInitialARGs(); _13 < _14.length; _13++) {
                var arg = _14[_13];
                var property = arg.getProperty();
                if (property && docker_1.Util.rangeEquals(property.getNameRange(), definitionRange)) {
                    for (var _15 = 0, _16 = dockerfile.getFROMs(); _15 < _16.length; _15++) {
                        var from = _16[_15];
                        for (var _17 = 0, _18 = from.getVariables(); _17 < _18.length; _17++) {
                            var variable = _18[_17];
                            if (variable.getName() === definition) {
                                highlights.push(vscode_languageserver_types_1.DocumentHighlight.create(variable.getNameRange(), vscode_languageserver_types_1.DocumentHighlightKind.Read));
                            }
                        }
                    }
                }
            }
        }
        return highlights;
    };
    return DockerHighlight;
}());
exports.DockerHighlight = DockerHighlight;
